using System;
using System.Drawing;
using System.Windows.Forms;

namespace AudioReverb
{
    public class WaveformPainter : Control
    {
        private float[] waveformData;
        private readonly Pen waveformPen;
        private readonly BufferedGraphicsContext context;
        private BufferedGraphics grafx;

        public WaveformPainter()
        {
            SetStyle(ControlStyles.AllPaintingInWmPaint | 
                    ControlStyles.UserPaint | 
                    ControlStyles.OptimizedDoubleBuffer, true);
            waveformPen = new Pen(Color.Lime, 1);
            context = BufferedGraphicsManager.Current;
            UpdateGraphics();
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            UpdateGraphics();
            base.OnSizeChanged(e);
        }

        private void UpdateGraphics()
        {
            if (Width > 0 && Height > 0)
            {
                grafx?.Dispose();
                grafx = context.Allocate(CreateGraphics(), ClientRectangle);
            }
        }

        public void UpdateWaveform(float[] data)
        {
            waveformData = data;
            Invalidate();
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            if (grafx == null || waveformData == null) return;

            grafx.Graphics.Clear(BackColor);
            float centerY = Height / 2f;
            float scaleY = Height / 2f;

            for (int i = 1; i < waveformData.Length; i++)
            {
                float x1 = (i - 1) * Width / (float)waveformData.Length;
                float x2 = i * Width / (float)waveformData.Length;
                float y1 = centerY + (waveformData[i - 1] * scaleY);
                float y2 = centerY + (waveformData[i] * scaleY);

                grafx.Graphics.DrawLine(waveformPen, x1, y1, x2, y2);
            }

            grafx.Render(e.Graphics);
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                waveformPen.Dispose();
                grafx?.Dispose();
            }
            base.Dispose(disposing);
        }
    }
} 