﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;

public class FormResizer
{
    private Size originalFormSize;
    private Dictionary<Control, ControlInfo> originalControlInfo;

    public FormResizer(Form form)
    {
        originalFormSize = form.Size;
        originalControlInfo = new Dictionary<Control, ControlInfo>();

        CollectControlInfo(form);
    }

    private void CollectControlInfo(Control control)
    {
        foreach (Control child in control.Controls)
        {
            originalControlInfo[child] = new ControlInfo
            {
                Bounds = child.Bounds,
                FontSize = child.Font.Size
            };

            // Recursively collect info for child controls
            if (child.HasChildren)
            {
                CollectControlInfo(child);
            }
        }
    }

    public void ResizeControls(Form form)
    {
        if (originalControlInfo == null) return;

        if (form.WindowState == FormWindowState.Maximized)
        {
            // Hide the form
            form.Visible = false;

            // Adjust the size and position of all controls
            foreach (Control control in form.Controls)
            {
                ResizeControl(control, form);
            }

            // Use Timer to delay displaying the form
            Timer timer = new Timer
            {
                Interval = 1 // 0.001 seconds
            };
            timer.Tick += (s, args) =>
            {
                timer.Stop();
                form.Visible = true;
            };
            timer.Start();
        }
        else
        {
            // Adjust the size and position of all controls
            foreach (Control control in form.Controls)
            {
                ResizeControl(control, form);
            }
        }
    }

    private void ResizeControl(Control control, Form form)
    {
        if (originalControlInfo.TryGetValue(control, out ControlInfo originalInfo))
        {
            float xRatio = (float)form.Width / originalFormSize.Width;
            float yRatio = (float)form.Height / originalFormSize.Height;

            int newX = (int)(originalInfo.Bounds.X * xRatio);
            int newY = (int)(originalInfo.Bounds.Y * yRatio);
            int newWidth = (int)(originalInfo.Bounds.Width * xRatio);
            int newHeight = (int)(originalInfo.Bounds.Height * yRatio);

            control.SetBounds(newX, newY, newWidth, newHeight);

            // Adjust font size
            float newFontSize = originalInfo.FontSize * Math.Min(xRatio, yRatio);
            control.Font = new Font(control.Font.FontFamily, newFontSize, control.Font.Style);

            // Recursively resize child controls
            if (control.HasChildren)
            {
                foreach (Control child in control.Controls)
                {
                    ResizeControl(child, form);
                }
            }
        }
    }
}

public class ControlInfo
{
    public Rectangle Bounds { get; set; }
    public float FontSize { get; set; }
}
